import { TraceDataSchema, TraceQueryFilterSchema, } from '@genkit-ai/tools-common';
import { logger } from '@genkit-ai/tools-common/utils';
import express from 'express';
export { LocalFileTraceStore } from './localFileTraceStore.js';
export { TraceQuerySchema } from './types';
let server;
export async function startTelemetryServer(params) {
    await params.traceStore.init();
    const api = express();
    api.use(express.json({ limit: params.maxRequestBodySize ?? '30mb' }));
    api.get('/api/__health', async (_, response) => {
        response.status(200).send('OK');
    });
    api.get('/api/traces/:traceId', async (request, response, next) => {
        try {
            const { traceId } = request.params;
            response.json(await params.traceStore.load(traceId));
        }
        catch (e) {
            next(e);
        }
    });
    api.post('/api/traces', async (request, response, next) => {
        try {
            const traceData = TraceDataSchema.parse(request.body);
            await params.traceStore.save(traceData.traceId, traceData);
            response.status(200).send('OK');
        }
        catch (e) {
            next(e);
        }
    });
    api.get('/api/traces', async (request, response, next) => {
        try {
            const { limit, continuationToken, filter } = request.query;
            response.json(await params.traceStore.list({
                limit: limit ? Number.parseInt(limit.toString()) : 10,
                continuationToken: continuationToken
                    ? continuationToken.toString()
                    : undefined,
                filter: filter
                    ? TraceQueryFilterSchema.parse(JSON.parse(filter))
                    : undefined,
            }));
        }
        catch (e) {
            next(e);
        }
    });
    api.use((err, req, res, next) => {
        logger.error(err.stack);
        const error = err;
        const { message, stack } = error;
        const errorResponse = {
            code: 13,
            message,
            details: {
                stack,
                traceId: err.traceId,
            },
        };
        res.status(500).json(errorResponse);
    });
    server = api.listen(params.port, () => {
        logger.info(`Telemetry API running on http://localhost:${params.port}`);
    });
    server.on('error', (error) => {
        logger.error(error);
    });
    process.on('SIGTERM', async () => await stopTelemetryApi());
}
export async function stopTelemetryApi() {
    await Promise.all([
        new Promise((resolve) => {
            if (server) {
                server.close(() => {
                    logger.debug('Telemetry API has succesfully shut down.');
                    resolve();
                });
            }
            else {
                resolve();
            }
        }),
    ]);
}
//# sourceMappingURL=index.js.map